rooms = ["room(one)", "room(two)", "room(three)"]
boxes = ["box(red)", "box(blue)", "box(green)"]
robot = "robot"

AT = "At"  # predicate name (just a string)


def at(thing, room):
    return (AT, thing, room)


class Actions:
    @staticmethod
    def goto(destination):
        # Precondition: can't "goto" where you already are
        def is_applicable(state):
            return at(robot, destination) not in state

        def apply(state):
            new_state = state.copy()

            # Remove old At(robot, room_?) facts
            for r in rooms:
                fact = at(robot, r)
                if fact in new_state:
                    new_state.remove(fact)

            # Add new At(robot, destination)
            new_state.add(at(robot, destination))
            return new_state

        apply.name = "goto(" + destination + ")"
        apply.is_applicable = is_applicable
        return apply

    @staticmethod
    def carry(unique_box, destination):
        # Precondition: robot and box are in the same room
        def is_applicable(state):
            for r in rooms:
                if at(robot, r) in state and at(unique_box, r) in state:
                    return True
            return False

        def apply(state):
            new_state = state.copy()

            # Find the current shared room, then move both robot and box there
            current_room = None
            for r in rooms:
                if at(robot, r) in new_state and at(unique_box, r) in new_state:
                    current_room = r
                    break

            if current_room is None:
                return new_state  # should not happen if precondition checked

            # Remove old robot + box locations
            for r in rooms:
                new_state.discard(at(robot, r))
                new_state.discard(at(unique_box, r))

            # Add new locations
            new_state.add(at(robot, destination))
            new_state.add(at(unique_box, destination))
            return new_state

        apply.name = "carry(" + unique_box + ", " + destination + ")"
        apply.is_applicable = is_applicable
        return apply


def breadth_first_search(start_state, goal_facts, actions, max_depth):
    queue = [(start_state, [])]  # (state, path)

    while queue:
        state, path = queue.pop(0)

        # Goal test: all goal facts must be present (predicate-generic)
        if goal_facts.issubset(state):
            return path

        if len(path) >= max_depth:
            continue

        for action in actions:
            if action.is_applicable(state):
                next_state = action(state)
                queue.append((next_state, path + [action.name]))

    return None


# Ground the domain.
# This is where we go from "carry some hypothetical box to some hypothetical
# location" to "carry the red box to room one" and "carry the red box to room
# two" and "carry..."
grounded_actions = []
for r in rooms:
    grounded_actions.append(Actions.goto(r))
for b in boxes:
    for r in rooms:
        grounded_actions.append(Actions.carry(b, r))


# Start state s0
s0 = {
    at("box(red)", "room(one)"),
    at("box(blue)", "room(one)"),
    at("box(green)", "room(one)"),
    at(robot, "room(two)"),
}

# Goal: At(box(red), room(two))
goal = {at("box(red)", "room(two)")}

plan = breadth_first_search(s0, goal, grounded_actions, max_depth=5)

print("Goal:", goal)
print("Plan:", plan)
